#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "pgm.h"
#include "ppm.h"

PPM	*ppm_alloc(int r, int c)
{
	PPM *img = NULL;

  img = (PPM *)malloc(sizeof(PPM));

  if(!img) {
    fprintf(stderr,"WARNING: could not allocate memory for img\n");
  }

  img->name = NULL;
  img->rows = r;
  img->cols = c;
  img->maxc = 255;
  img->magic = (char *)malloc(3*sizeof(char));
  strcpy(img->magic,"P6");
  img->rdata = (float *)malloc((r*c)*sizeof(float));
  img->gdata = (float *)malloc((r*c)*sizeof(float));
  img->bdata = (float *)malloc((r*c)*sizeof(float));

  if(!img->rdata || !img->gdata || !img->bdata) {
    fprintf(stderr,"WARNING: could not allocate memory for img data\n");
  }

  return img;
}

void     ppm_free(PPM **img)
{
  // free internals, anything that got allocated
  if((*img)->name)
    free((*img)->name);

  if((*img)->magic)
    free((*img)->magic);

  if((*img)->rdata)
    free((*img)->rdata);

  if((*img)->gdata)
    free((*img)->gdata);

  if((*img)->bdata)
    free((*img)->bdata);

  free((*img));
  (*img) = NULL;
}

PPM     *ppm_read(const char *file)
{
	int		i,j;
	int		rows,cols,maxc;
	char		c;
	PPM 		*img = NULL;
	FILE		*in = NULL;
	unsigned char	r,g,b;

  // open file
  in = fopen(file,"r");
  if(!in) {
    fprintf(stderr,"Error in opening %s\n", file);
    return img;
  }

  // read input
  if( ((c=fgetc(in))=='P') && ((c=fgetc(in))=='6') && ((c=fgetc(in))=='\n')) {

    // ppm images can have # comments after first line
    if((c=fgetc(in))=='#')
      do {
        c=fgetc(in);
      } while(c!='\n');
    else
      ungetc(c,in);

    // read width, height
    if(fscanf(in,"%u %u",&cols,&rows) < 2)
      fprintf(stderr,"Error getting width, height from %s\n", file);

    // read maxc
    if( (fscanf(in,"%u\n",&maxc) < 1) || maxc != 255 )
      fprintf(stderr,"Error getting maxc from %s\n", file);

    img = ppm_alloc(rows,cols);

    if(img) {
      // read in data
      for(i=0;i<rows;i++) {
        for(j=0;j<cols;j++) {
          if( (fread(&r,sizeof(unsigned char),1,in) < 1) || feof(in) )
            fprintf(stderr,"Error getting r from %s\n", file);
          if( (fread(&g,sizeof(unsigned char),1,in) < 1) || feof(in) )
            fprintf(stderr,"Error getting g from %s\n", file);
          if( (fread(&b,sizeof(unsigned char),1,in) < 1) || feof(in) )
            fprintf(stderr,"Error getting b from %s\n", file);

          img->rdata[i*img->cols + j] = (float)r/(float)maxc;
          img->gdata[i*img->cols + j] = (float)g/(float)maxc;
          img->bdata[i*img->cols + j] = (float)b/(float)maxc;

        }
      }
    }

  } else {
    fprintf(stderr,"Unknown character in %s\n", file);
  }

  return img;
}

int      ppm_write(PPM *img, const char *file)
{
	int		i,j;
	FILE		*out;
	unsigned char	r,g,b;

  // open output file for writing
  out = fopen(file,"w");
  if(!out) {
    printf("Error in opening %s\n", file);
    return 1;
  }

  // write output
  fprintf(out,"%s\n%d %d\n%d\n",img->magic,img->rows,img->cols,img->maxc);
  for(i=0;i<img->rows;i++) {
    for(j=0;j<img->cols;j++) {
      r = (unsigned char)(img->rdata[i*img->cols + j] * (float)img->maxc);
      g = (unsigned char)(img->gdata[i*img->cols + j] * (float)img->maxc);
      b = (unsigned char)(img->bdata[i*img->cols + j] * (float)img->maxc);
      fwrite(&r,sizeof(unsigned char),1,out);
      fwrite(&g,sizeof(unsigned char),1,out);
      fwrite(&b,sizeof(unsigned char),1,out);
    }
  }
  fflush(out);

  fclose(out);

  return 0;
}

void     ppm_white(PPM *img)
{
	int i,j;

  for(i=0; i<img->rows; i++) {
    for(j=0; j<img->cols; j++) {
      img->rdata[i*img->cols + j] = 1.0;
      img->gdata[i*img->cols + j] = 1.0;
      img->bdata[i*img->cols + j] = 1.0;
    }
  }
}

PGM     *ppm_togrey(PPM *cimg)
{
	int	i,j;
	PGM	*gimg=NULL;
	float	r,g,b,l;

  gimg = pgm_alloc(cimg->rows, cimg->cols);

  for(i=0; i<cimg->rows; i++) {
    for(j=0; j<cimg->cols; j++) {
      r = cimg->rdata[i*cimg->cols + j];
      g = cimg->gdata[i*cimg->cols + j];
      b = cimg->bdata[i*cimg->cols + j];
      l = 0.21 * r + 0.72 * g + 0.07 * b;
      gimg->data[i*gimg->cols + j] = l;
    }
  }

  return gimg;
}
