#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "defs.h"
#include "filters.h"
#include "pgm.h"
#include "arrays.h"

PGM	*pgm_alloc(int r, int c)
{
	PGM *img = NULL;

  // allocate struct
  img = (PGM *)malloc(sizeof(PGM));

  if(!img) {
    fprintf(stderr,"WARNING: could not allocate memory for img\n");
  }

  // initialize struct contents
  img->name = NULL;
  img->rows = r;
  img->cols = c;
  img->maxc = 255;
  img->magic = (char *)malloc(3*sizeof(char));
  strcpy(img->magic,"P5");
  img->data = (double *)malloc((r*c)*sizeof(double));

  // if we couldn't allocate room for r*c data
  if(!img->data) {
    fprintf(stderr,"WARNING: could not allocate memory for img data\n");
  }

  return img;
}

PGM     *pgm_read(const char *file)
{
	int		i,j;
	int		rows,cols,maxc;
	char		c;
	PGM 		*img = NULL;
	FILE		*in = NULL;
	unsigned char	data;

  // open file
  in = fopen(file,"r");
  if(!in) {
    fprintf(stderr,"Error in opening %s\n", file);
    return img;
  }

  // read input
  if( ((c=fgetc(in))=='P') && ((c=fgetc(in))=='5') && ((c=fgetc(in))=='\n')) {

    // ppm images can have # comments after first line
    if((c=fgetc(in))=='#')
      do {
        c=fgetc(in);
      } while(c!='\n');
    else
      ungetc(c,in);

    // read width, height
    if(fscanf(in,"%u %u",&cols,&rows) < 2)
      fprintf(stderr,"Error getting width, height from %s\n", file);

    // read maxc
    if( (fscanf(in,"%u\n",&maxc) < 1) || maxc != 255 )
      fprintf(stderr,"Error getting maxc from %s\n", file);

    img = pgm_alloc(rows,cols);

    if(img) {
      // read in data
      for(i=0;i<rows;i++) {
        for(j=0;j<cols;j++) {
          if( (fread(&data,sizeof(unsigned char),1,in) < 1) || feof(in) )
            fprintf(stderr,"Error getting r from %s\n", file);

          img->data[i*img->cols + j] = (double)data/(double)maxc;

        }
      }
    }

  } else {
    fprintf(stderr,"Unknown character in %s\n", file);
  }

  return img;
}

PGM     *pgm_copy(PGM *img)
{
	int	r = img->rows, c = img->cols;
	PGM 	*dimg = NULL;

  dimg = pgm_alloc(r,c);

  if(img->name) {
    dimg->name = (char *)malloc(strlen(img->name+1)*sizeof(char));
    strcpy(dimg->name,img->name);
  }

  memcpy(dimg->data,img->data,(r*c)*sizeof(double));

  return(dimg);
}

void     pgm_free(PGM **img)
{
  // free internals, anything that got allocated
  if((*img)->name)
    free((*img)->name);

  if((*img)->magic)
    free((*img)->magic);

  if((*img)->data)
    free((*img)->data);

  // free (pointer to) struct
  free((*img));
  (*img) = NULL;
}

int      pgm_write(PGM *img, const char *file)
{
	int		i,j;
 	FILE		*out;
	unsigned char	grey;

  // open output file for writing
  out = fopen(file,"w");
  if(!out) {
    printf("Error in opening %s\n", file);
    return 1;
  }

  // write output
  fprintf(out,"%s\n%d %d\n%d\n",img->magic,img->rows,img->cols,img->maxc);
  for(i=0;i<img->rows;i++) {
    for(j=0;j<img->cols;j++) {
      grey = (unsigned char)(img->data[i*img->cols + j] * (double)img->maxc);
      fwrite(&grey,sizeof(unsigned char),1,out);
    }
  }
  fflush(out);

  fclose(out);

  return 0;
}

void     pgm_normalize(PGM *img)
{
  darr_normalize(img->data,img->rows,img->cols);
}

PGM *pgm_conv(PGM *img,FLTR *fp)
{
	int	i,j,m,r,c;
	PGM    *cimg=NULL, *timg=NULL;

  // copy
  cimg = pgm_alloc(img->rows, img->cols);
  timg = pgm_alloc(img->rows, img->cols);

  // zero out
  memset(cimg->data, 0, r*c*sizeof(double));
  memset(timg->data, 0, r*c*sizeof(double));

  // copy name
  if(img->name) {
    cimg->name = (char *)malloc(strlen(img->name+1)*sizeof(char));
    strcpy(cimg->name,img->name);
  }

  r = img->rows;  c = img->cols;

  for(i=0;i<r;i++) {
    for(j=0;j<c;j++) {
      for(m=0;m<fp->len;m++) {
        timg->data[M2A(i,j,r,c)] += fp->dx[m]*img->data[M2A(i,j+m,r,c)];
      }
    }
  }

  for(i=0;i<r;i++) {
    for(j=0;j<c;j++) {
      for(m=0;m<fp->len;m++) {
        cimg->data[M2A(i,j,r,c)] += fp->dy[m]*timg->data[M2A(i+m,j,r,c)];
      }
    }
  }

  return cimg;
}
